/*
 * Copyright (c) 2018 Particle Industries, Inc.  All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

#include <FreeRTOS.h>
#include <task.h>
#include "service_debug.h"
#include "hal_event.h"
#include "concurrent_hal.h"
#include <atomic>

namespace {

std::atomic_bool sRestoreIdleThreadPriority(false);

} // anonymous

extern "C" {

#ifdef DEBUG_BUILD
void vApplicationStackOverflowHook(TaskHandle_t, char*) {
    PANIC(StackOverflow, "Stack overflow detected");
}
#endif /* DEBUG_BUILD */

void vApplicationMallocFailedHook(size_t xWantedSize) {
    hal_notify_event(HAL_EVENT_OUT_OF_MEMORY, xWantedSize, 0);
}

#if configSUPPORT_STATIC_ALLOCATION == 1

/* configSUPPORT_STATIC_ALLOCATION is set to 1, so the application must provide an
implementation of vApplicationGetIdleTaskMemory() to provide the memory that is
used by the Idle task. */
void vApplicationGetIdleTaskMemory( StaticTask_t **ppxIdleTaskTCBBuffer,
                                    StackType_t **ppxIdleTaskStackBuffer,
                                    uint32_t *pulIdleTaskStackSize )
{
/* If the buffers to be provided to the Idle task are declared inside this
function then they must be declared static - otherwise they will be allocated on
the stack and so not exists after this function exits. */
static StaticTask_t xIdleTaskTCB;
static StackType_t uxIdleTaskStack[ configMINIMAL_STACK_SIZE ];

    /* Pass out a pointer to the StaticTask_t structure in which the Idle task's
    state will be stored. */
    *ppxIdleTaskTCBBuffer = &xIdleTaskTCB;

    /* Pass out the array that will be used as the Idle task's stack. */
    *ppxIdleTaskStackBuffer = uxIdleTaskStack;

    /* Pass out the size of the array pointed to by *ppxIdleTaskStackBuffer.
    Note that, as the array is necessarily of type StackType_t,
    configMINIMAL_STACK_SIZE is specified in words, not bytes. */
    *pulIdleTaskStackSize = configMINIMAL_STACK_SIZE;
}
/*-----------------------------------------------------------*/

/* configSUPPORT_STATIC_ALLOCATION and configUSE_TIMERS are both set to 1, so the
application must provide an implementation of vApplicationGetTimerTaskMemory()
to provide the memory that is used by the Timer service task. */
void vApplicationGetTimerTaskMemory( StaticTask_t **ppxTimerTaskTCBBuffer,
                                     StackType_t **ppxTimerTaskStackBuffer,
                                     uint32_t *pulTimerTaskStackSize )
{
/* If the buffers to be provided to the Timer task are declared inside this
function then they must be declared static - otherwise they will be allocated on
the stack and so not exists after this function exits. */
static StaticTask_t xTimerTaskTCB;
static StackType_t uxTimerTaskStack[ configTIMER_TASK_STACK_DEPTH ];

    /* Pass out a pointer to the StaticTask_t structure in which the Timer
    task's state will be stored. */
    *ppxTimerTaskTCBBuffer = &xTimerTaskTCB;

    /* Pass out the array that will be used as the Timer task's stack. */
    *ppxTimerTaskStackBuffer = uxTimerTaskStack;

    /* Pass out the size of the array pointed to by *ppxTimerTaskStackBuffer.
    Note that, as the array is necessarily of type StackType_t,
    configTIMER_TASK_STACK_DEPTH is specified in words, not bytes. */
    *pulTimerTaskStackSize = configTIMER_TASK_STACK_DEPTH;
}
#endif /* configSUPPORT_STATIC_ALLOCATION == 1 */

void vApplicationTaskDeleteHook(void* pvTaskToDelete, volatile BaseType_t* pxPendYield) {
    (void)pvTaskToDelete;
    (void)pxPendYield;

    // NOTE: this hook is executed within a critical section

    sRestoreIdleThreadPriority = true;

    // Temporarily raise IDLE thread priority to (configMAX_PRIORITIES - 1) (maximum)
    // to give it some processing time to clean up the deleted task resources.
    vTaskPrioritySet(xTaskGetIdleTaskHandle(), configMAX_PRIORITIES - 1);

    // Immediately request the scheduler to yield to now higher priority IDLE thread
    *pxPendYield = pdTRUE;
}

void vApplicationIdleHook(void) {
    if (sRestoreIdleThreadPriority.exchange(false)) {
        // Restore IDLE thread priority back to the default one
        vTaskPrioritySet(nullptr, tskIDLE_PRIORITY);
    }
}

} // extern "C"
